Attribute VB_Name = "WaveAnalysis"
Option Explicit
'---------------WAVE ANALYSIS FUNCTIONS-------------------
'   CreateSineWave():   This routine generates a sine wave (one full period
'                       will fit into the NumberOfSamples) and stores it in
'                       a WaveData() array.
'
'   CreateSquareWave(): This routine isn't called directly, but it might
'                       be useful for testing the response of a system without
'                       a sine wave.
'
'   GetPhase(): This function returns a string containing a valid Excel expression
'               for the phase shift of the current wave.
'
'   FindLocalMaxima():  This function runs through data and finds all local maxima
'                       (hopefully).  Called by GetPhase().
'
'   MaximaCheckRoutine():   The routine called by FindLocalMaxima() to actually
'                           check that a point is a maximum.  Called by FindLocalMaxima().
'
'   IsArrayTrue():      Takes a boolean array, returns true if ALL elements are TRUE.
'                       Used by MaximaCheckRoutine().
'
'Type wave includes all common characteristics of a wave--for function calls.
Public Type Wave
    Amplitude As Single
    Cycles As Single
    DutyCycle As Single
    Frequency As Single
    Phase As Single
    Offset As Single
End Type
Public Sub CreateSineWave(OutputWave As Wave, OutputWaveData() As Single, NumberOfSamples As Integer)
    'This function takes an array of variable length and fills it with data from a sine wave
    'using supplied values for Amplitude, Offset, and as many Cycles as requested.
    
    Const PI# = 3.14159265358979
    Dim theta#
    Dim i As Integer
    
    theta# = (2# * PI) / NumberOfSamples     'Our incremental value.
    
    With OutputWave
        For i = 0 To NumberOfSamples - 1
            OutputWaveData(i) = .Amplitude * Sin(.Cycles * theta# * i) + .Offset
        Next i
    End With
    
End Sub
Public Sub CreateSquareWave(OutputWave As Wave, OutputWaveData() As Single, NumberOfSamples As Integer)
    'Just included as an example, in case we wish to see square wave response
    'of a circuit.
    
    Dim CycleNum As Integer, OffsetInCycle As Integer, tmp As Integer
    Dim UpTime As Single, DownTime As Single
    
    OutputWave.DutyCycle = 0.5
    
    UpTime = OutputWave.DutyCycle
    DownTime = 1 - OutputWave.DutyCycle
        
    With OutputWave
    For CycleNum = 0 To NumberOfSamples - 1 Step (NumberOfSamples \ OutputWave.Cycles)
        'First, we fill the array with the bottom portion.
        For OffsetInCycle = 0 To CInt(DownTime * (NumberOfSamples \ OutputWave.Cycles))
            OutputWaveData(CycleNum + OffsetInCycle) = .Offset
        Next OffsetInCycle
        
        tmp = OffsetInCycle
        
        'Then, we fill it with the top portion and repeat to however many Cycles there are.
        For OffsetInCycle = tmp To CInt(UpTime * (NumberOfSamples \ OutputWave.Cycles))
            OutputWaveData(CycleNum + OffsetInCycle) = .Offset + .Amplitude
        Next OffsetInCycle
    Next CycleNum
    End With
    
    'Use this algorithm to get the maximum frequency out of the card.
    'It just creates a wave of alternating high and low every sample
    '(duty cycle .5)
    
    'With OutputWave
    '    For i = 0 To NumberOfSamples - 1
    '        If (i Mod 2) = 0 Then
    '            OutputWaveData(i) = .Amplitude + .Offset
    '        Else
    '            OutputWaveData(i) = .Offset
    '        End If
    '    Next i
    'End With
    
End Sub

Public Function GetPhase(InputWaveData() As Single, OutputWaveData() As Single, w1 As Wave, ByVal NumberOfSamples As Integer) As String
    'Returns a *STRING* containing a formula to go into Excel which calculates the shift
    'between InputWaveData() and OutputWaveData().
    
    'N.B.: This function does not have good accuracy.  Anywhere from 5 to 10 degrees
    'error is common.  This is especially true when sampling at low frequencies.  If FindLocalMaxima()
    'misses a peak, then it will be off by ~360 degrees.  But it's good for a basic routine.
    
    Dim NumberAI As Integer, NumberAO As Integer, i As Integer
    Dim RunningTotal As Single, AvgDif As Single
    
    ReDim MaximaAO(0 To UBound(OutputWaveData()) - 1) As Single
    ReDim MaximaAI(0 To UBound(InputWaveData()) - 1) As Single
    ReDim Difference(0 To UBound(OutputWaveData()) - 1) As Single
    
    'First, we need the Maxima.
    NumberAI = FindLocalMaxima(MaximaAI(), InputWaveData())   'Use AI Wave
    NumberAO = FindLocalMaxima(MaximaAO(), OutputWaveData())  'Use AO wave
    
    If (NumberAI = 0) Or (NumberAO = 0) Then    'Oops, no maxima found.
        GetPhase = "#ERR"
    Else
        'Please see comment in readme.txt for the reasoning behind this for loop.
        For i = 0 To UBound(MaximaAI())
            MaximaAI(i) = MaximaAI(i) - 1
        Next i
        
        If UBound(MaximaAI()) = UBound(MaximaAO()) Then
            'Calculate the average phase over several cycles if we can.
            ReDim Difference(0 To UBound(MaximaAI()))
            For i = 0 To UBound(MaximaAI())
                Difference(i) = MaximaAI(i) - MaximaAO(i)
            Next i
        
            For i = 0 To UBound(Difference())
                RunningTotal = RunningTotal + Difference(i)
            Next i
            
            AvgDif = RunningTotal / (UBound(Difference()) + 1)
            
            Select Case 360 * AvgDif / (NumberOfSamples / w1.Cycles)
                'We want our values to fall within -180 < phase < 180
                Case Is < -180
                    GetPhase = "=360*((" & AvgDif & ")/($N$20/$N$17))+360"
                Case Is > 180
                    GetPhase = "=360*((" & AvgDif & ")/($N$20/$N$17))-360"
                Case Else
                    GetPhase = "=360*((" & AvgDif & ")/($N$20/$N$17))"
            End Select
        
        Else
            'Otherwise, just use the first sample.
            '(Remember, K20 and K17 contain NumberOfSamples and OutputWave.Frequency,
            'respectively).
            GetPhase = "=360*((" & MaximaAI(0) & "-" & MaximaAO(0) & ")/($N$20/$N$17))"
            
            Select Case 360 * (MaximaAI(0) - MaximaAO(0)) / (NumberOfSamples / w1.Cycles)
                Case Is < -180
                    GetPhase = "=360*((" & MaximaAI(0) & "-" & MaximaAO(0) & ")/($N$20/$N$17))+360"
                Case Is > 180
                    GetPhase = "=360*((" & MaximaAI(0) & "-" & MaximaAO(0) & ")/($N$20/$N$17))-360"
                Case Else
                    GetPhase = "=360*((" & MaximaAI(0) & "-" & MaximaAO(0) & ")/($N$20/$N$17))"
            End Select
        
        End If
    
        
    End If
End Function

Private Function FindLocalMaxima(Maxima() As Single, WaveData() As Single) As Integer
    'FindLocalMaxima takes an array with the same bounds as Output/InputWaveData array.
    'When this function is complete, Maxima will contain a list of sample numbers
    'which correspond to local maxima in WaveData(), having as many elements as
    'there are maxima in the wave.
    
    'FindLocalMaxima() returns the number of elements it entered into Maxima().
    
    'Maxima() contains the index number of the local maxima of Output/InputWaveData().
    'For example, if inputwavedata() were to peak at sample number 542 and 897,
    'then Maxima(0) would = 542 and Maxima(1) would = 897.  FindLocalMaxima()
    'would return 2.
    
    Const SampleTolerance As Integer = 3    'Defines how many points away the
                                            'MaximaCheckRoutine will go to ensure
                                            'that the current data point is > nearby ones.
    
    Dim i As Integer, NumberReturned As Integer
    
    'The for loop ensures that we don't miss any points simply because we had
    'the wrong SampleTolerance setting.
    
    'This function returns the first set of maxima it gets--even if MaximaCheckRoutine would
    'have returned correct values for a smaller SampleTolerance.
    For i = SampleTolerance To 1 Step -1
        NumberReturned = MaximaCheckRoutine(Maxima(), i, WaveData())
        If NumberReturned Then  'Found some
            ReDim Preserve Maxima(0 To NumberReturned - 1)
            Exit For
        End If
    Next i
  
    FindLocalMaxima = NumberReturned
End Function
Private Function MaximaCheckRoutine(Maxima() As Single, ByVal SampleTolerance As Integer, WaveData() As Single) As Integer
    'This function takes an array and returns with values corresponding to the index
    'numbers at which it found maxima.
    
    'The function itself returns the number of values entered into Maxima.
    
    'This function most easily finds maxima of the form
    '(1, 1.25, 1.5, 1.25, 1, ...)
    
    'The SampleTolerance helps to account for sequences such as
    '(-1, -1.25, -1.125, -1.35, -1.5, ...), so that the -1.125 is not recorded
    'as a relative max (even though, strictly speaking, it is a relative max).
    
    'This algorithm is confused and will not find a maxima of the form:
    '(1, 1.25, 1.5, 1.5, 1.5, 1.25., 1, ...) (3 points with same value),
    'though it will find a maxima of the form
    '(1, 1.25, 1.5, 1.5, 1.25, 1, ...) (i.e., 2 points with same value).
    'Really, this sort of behavior is heavily modifyable, and is just a question of
    'how many similar values you want to trap.
    
    'If it does find a sequence of the latter type, it will place 3.5 (in the
    'example's case) into Maxima().
    
    'The algorithm is particularly bad at analyzing noisy signals.  The maxima analysis routine
    'does not account for sequences such as
    '(0, .25, .5, .49, .51, .75, 1) in the case that there is a lot of noise.  If the noise is localized,
    'then GetPhase calling it with large SampleTolerance should correct behavior.
    
    Dim i As Integer, j As Integer, k As Integer, counter As Integer
    Dim HalfwayBetween As Boolean
    ReDim TrueForThisSample(0 To SampleTolerance - 1) As Boolean
        
    For i = 1 To UBound(WaveData())
        k = 0
        If i > SampleTolerance And (UBound(WaveData()) - i > SampleTolerance) Then
            For j = 1 To SampleTolerance
                'Check several samples away, to make sure it's not just a data fluke.
                'e.g. (-1, -1.25, -1.125, -1.5, -1.75, -2, ...)
                If (WaveData(i + j) < WaveData(i)) And WaveData(i - j) < WaveData(i) _
                Then 'We've got a LocalMax!
                    TrueForThisSample(k) = True
                    k = k + 1
                End If
            Next j
                
        'Maybe we missed a local max because of a leveling out?
        '(e.g., 1, 1.25, 1.5, 1.5, 1.25, 1, ...)
        If (WaveData(i + 1) = WaveData(i)) Then 'Got a leveling out
                If WaveData(i + 2) < WaveData(i) And WaveData(i - 1) < WaveData(i) Then
                    For k = 0 To SampleTolerance - 1
                        TrueForThisSample(k) = True
                    Next k
                    HalfwayBetween = True
                    k = 0
                End If
            End If
        Else
            If i <= SampleTolerance Then     'We're at left edge of wave data
                For j = 1 To i
                    If (WaveData(i + j) < WaveData(i)) And (WaveData(i - j) < WaveData(i)) Then  'we've got a localmax
                        TrueForThisSample(k) = True
                        k = k + 1
                    End If
                Next j
            Else    'ubound(WaveData())-i > SampleTolerance    'Right edge of wave data
                For j = 1 To UBound(WaveData()) - i
                    If (WaveData(i + j) < WaveData(i)) And (WaveData(i - j) < WaveData(i)) Then 'we've got a localmax
                        TrueForThisSample(k) = True
                        k = k + 1
                    End If
                Next j
            End If
        End If
            
        If IsArrayTrue(TrueForThisSample()) Then
            If HalfwayBetween Then
                Maxima(counter) = i + 0.5!
                HalfwayBetween = False
                counter = counter + 1
            Else
                Maxima(counter) = i
                counter = counter + 1
            End If
        End If
            
        'Reset array to false.
        For j = 0 To UBound(TrueForThisSample())
            TrueForThisSample(j) = False
        Next j
            
    Next i
    
    For i = 0 To counter - 1
        'Correct for being a sample off.  That is, array goes from 0 to NumberOfSamples-1,
        'but Real sample numbers go from 1 to NumberOfSamples.
        Maxima(i) = Maxima(i) + 1
    Next i
    
    MaximaCheckRoutine = counter
End Function

Private Function IsArrayTrue(BoolArray() As Boolean) As Boolean
    'Passed Array must be true AT EVERY ELEMENT for IsArrayTrue to return True.
    
    Dim i As Integer
    Dim RunningCount As Boolean

    For i = 0 To UBound(BoolArray())
        If BoolArray(i) Then
            RunningCount = True
        Else
            RunningCount = False
        End If
        
        If Not RunningCount Then
            Exit For
        End If
         
    Next i

    If i <> UBound(BoolArray()) + 1 Then      'Then we exited the loop early.
        IsArrayTrue = False
    Else
        IsArrayTrue = True
    End If
    
End Function
